<?php
declare(strict_types=1);
namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Support\DB;
use App\Support\Logger;
use App\Integrations\OpenPass\OpenPassClient;

final class AccountController
{
    private function authUserId(): int
    {
        $token = Request::header('Authorization');
        if (!$token) Response::json(['error' => 'unauthorized'], 401);
        $token = trim(str_replace('Bearer', '', $token));
        $row = DB::one("SELECT user_id FROM sessions WHERE token = ? AND expires_at > NOW() LIMIT 1", [$token]);
        if (!$row) Response::json(['error' => 'unauthorized'], 401);
        return (int)$row['user_id'];
    }

    public function list(): void
    {
        $uid = $this->authUserId();
        $rows = DB::all("SELECT id, provider, external_id, cvu, alias, status, created_at FROM accounts WHERE user_id = ? ORDER BY id DESC", [$uid]);
        Response::json(['ok' => true, 'accounts' => $rows]);
    }

    public function create(): void
    {
        $uid = $this->authUserId();
        $in = Request::json();

        // Minimal payload; you will extend with KYC fields, address, etc.
        $payload = [
            'user' => [
                'dni' => (string)($in['dni'] ?? ''),
                'email' => (string)($in['email'] ?? ''),
                'fullName' => (string)($in['fullName'] ?? ''),
            ],
        ];

        $client = new OpenPassClient();
        $res = $client->createAccount($payload);

        // Persist mapping (you must adjust based on actual OpenPass response fields)
        $externalId = (string)($res['id'] ?? $res['accountId'] ?? '');
        $cvu = (string)($res['cvu'] ?? '');
        $alias = (string)($res['alias'] ?? '');

        $accId = DB::insert("INSERT INTO accounts (user_id, provider, external_id, cvu, alias, status, created_at) VALUES (?, 'openpass', ?, ?, ?, 'active', NOW())", [
            $uid, $externalId, $cvu, $alias
        ]);

        Logger::audit('account_created', ['user_id' => $uid, 'account_id' => $accId, 'provider' => 'openpass']);

        Response::json(['ok' => true, 'account_id' => $accId, 'provider_response' => $res], 201);
    }
}
