<?php
declare(strict_types=1);
namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Support\DB;
use App\Support\Crypto;
use App\Support\Logger;

final class AuthController
{
    public function register(): void
    {
        $in = Request::json();
        $email = strtolower(trim((string)($in['email'] ?? '')));
        $password = (string)($in['password'] ?? '');
        $dni = preg_replace('/\D+/', '', (string)($in['dni'] ?? ''));

        if ($email === '' || $password === '' || $dni === '') {
            Response::json(['error' => 'validation', 'message' => 'email, password y dni son requeridos'], 422);
        }

        $exists = DB::one("SELECT id FROM users WHERE email = ? LIMIT 1", [$email]);
        if ($exists) Response::json(['error' => 'exists'], 409);

        $uid = DB::insert("INSERT INTO users (email, dni, password_hash, status, created_at) VALUES (?, ?, ?, 'active', NOW())", [
            $email, $dni, Crypto::hashPassword($password)
        ]);

        Logger::audit('user_registered', ['user_id' => $uid, 'email' => $email]);

        Response::json(['ok' => true, 'user_id' => $uid], 201);
    }

    public function login(): void
    {
        $in = Request::json();
        $email = strtolower(trim((string)($in['email'] ?? '')));
        $password = (string)($in['password'] ?? '');

        $row = DB::one("SELECT id, password_hash, status FROM users WHERE email = ? LIMIT 1", [$email]);
        if (!$row || $row['status'] !== 'active') Response::json(['error' => 'invalid_credentials'], 401);
        if (!Crypto::verifyPassword($password, $row['password_hash'])) Response::json(['error' => 'invalid_credentials'], 401);

        $token = Crypto::randomHex(32);
        DB::insert("INSERT INTO sessions (user_id, token, created_at, expires_at) VALUES (?, ?, NOW(), DATE_ADD(NOW(), INTERVAL 7 DAY))", [
            $row['id'], $token
        ]);

        Logger::audit('user_login', ['user_id' => $row['id']]);

        Response::json(['ok' => true, 'token' => $token]);
    }
}
