<?php
declare(strict_types=1);
namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Support\DB;
use App\Support\Logger;
use App\Integrations\OpenPass\OpenPassClient;

final class PaymentLinkController
{
    private function authUserId(): int
    {
        $token = Request::header('Authorization');
        if (!$token) Response::json(['error' => 'unauthorized'], 401);
        $token = trim(str_replace('Bearer', '', $token));
        $row = DB::one("SELECT user_id FROM sessions WHERE token = ? AND expires_at > NOW() LIMIT 1", [$token]);
        if (!$row) Response::json(['error' => 'unauthorized'], 401);
        return (int)$row['user_id'];
    }

    public function create(): void
    {
        $uid = $this->authUserId();
        $in = Request::json();
        $amount = (float)($in['amount'] ?? 0);
        $description = (string)($in['description'] ?? 'Pago');

        if ($amount <= 0) Response::json(['error' => 'validation'], 422);

        $pid = DB::insert("INSERT INTO payment_links (user_id, provider, amount, currency, description, status, created_at) VALUES (?, 'openpass', ?, 'ARS', ?, 'pending', NOW())", [
            $uid, $amount, $description
        ]);

        $payload = [
            'amount' => $amount,
            'currency' => 'ARS',
            'description' => $description,
            'referenceId' => (string)$pid,
            // You may need additional fields (payer, expiration, etc.) according to docs
        ];

        $client = new OpenPassClient();
        $res = $client->createPaymentLink($payload);

        $url = (string)($res['url'] ?? $res['paymentUrl'] ?? '');
        DB::exec("UPDATE payment_links SET provider_txn_id=?, url=?, status=?, updated_at=NOW(), provider_response=? WHERE id=?", [
            (string)($res['id'] ?? ''), $url, $url !== '' ? 'active' : 'pending',
            json_encode($res, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES), $pid
        ]);

        Logger::audit('payment_link_created', ['payment_link_id' => $pid, 'user_id' => $uid]);

        Response::json(['ok' => true, 'payment_link_id' => $pid, 'url' => $url, 'provider_response' => $res], 201);
    }
}
