<?php
declare(strict_types=1);
namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Support\DB;
use App\Support\Logger;
use App\Integrations\OpenPass\OpenPassClient;

final class TransferController
{
    private function authUserId(): int
    {
        $token = Request::header('Authorization');
        if (!$token) Response::json(['error' => 'unauthorized'], 401);
        $token = trim(str_replace('Bearer', '', $token));
        $row = DB::one("SELECT user_id FROM sessions WHERE token = ? AND expires_at > NOW() LIMIT 1", [$token]);
        if (!$row) Response::json(['error' => 'unauthorized'], 401);
        return (int)$row['user_id'];
    }

    public function create(): void
    {
        $uid = $this->authUserId();
        $in = Request::json();

        $amount = (float)($in['amount'] ?? 0);
        $to = (string)($in['to'] ?? ''); // CBU/CVU/ALIAS
        $fromAccountId = (int)($in['from_account_id'] ?? 0);

        if ($amount <= 0 || $to === '' || $fromAccountId <= 0) {
            Response::json(['error' => 'validation'], 422);
        }

        $acc = DB::one("SELECT * FROM accounts WHERE id = ? AND user_id = ? LIMIT 1", [$fromAccountId, $uid]);
        if (!$acc) Response::json(['error' => 'account_not_found'], 404);

        // Create local transfer record (pending)
        $tid = DB::insert("INSERT INTO transfers (user_id, account_id, provider, amount, currency, to_ref, status, created_at) VALUES (?, ?, 'openpass', ?, 'ARS', ?, 'pending', NOW())", [
            $uid, $fromAccountId, $amount, $to
        ]);

        $payload = [
            'from' => [
                'accountId' => $acc['external_id'],
                'cvu' => $acc['cvu'],
            ],
            'to' => [
                'destination' => $to,
            ],
            'amount' => $amount,
            'currency' => 'ARS',
            'referenceId' => (string)$tid,
        ];

        $client = new OpenPassClient();
        $res = $client->createTransfer($payload);

        $providerTxnId = (string)($res['id'] ?? $res['transferId'] ?? '');
        $newStatus = $providerTxnId !== '' ? 'submitted' : 'pending';

        DB::exec("UPDATE transfers SET provider_txn_id = ?, status = ?, updated_at = NOW(), provider_response = ? WHERE id = ?", [
            $providerTxnId, $newStatus, json_encode($res, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES), $tid
        ]);

        Logger::audit('transfer_created', ['transfer_id' => $tid, 'user_id' => $uid, 'status' => $newStatus]);

        Response::json(['ok' => true, 'transfer_id' => $tid, 'provider_response' => $res], 201);
    }
}
