<?php
declare(strict_types=1);
namespace App\Controllers;

use App\Http\Request;
use App\Http\Response;
use App\Support\DB;
use App\Support\Logger;
use App\Config;
use App\Support\Crypto;

final class WebhookController
{
    public function openpass(): void
    {
        $raw = file_get_contents('php://input') ?: '';
        $sig = (string)(Request::header('X-Signature') ?? '');

        // Optional HMAC validation (you can align header name + method with OpenPass specs)
        $secret = Config::webhooks()['openpass_secret'] ?? '';
        if ($secret !== '') {
            $expected = Crypto::hmac($raw, $secret);
            if (!hash_equals($expected, $sig)) {
                Logger::error('Webhook signature mismatch', ['expected' => $expected, 'received' => $sig]);
                Response::json(['error' => 'invalid_signature'], 401);
            }
        }

        $event = json_decode($raw, true);
        if (!is_array($event)) $event = ['raw' => $raw];

        // Store raw event
        DB::insert("INSERT INTO webhook_events (provider, event_type, payload, received_at) VALUES ('openpass', ?, ?, NOW())", [
            (string)($event['type'] ?? $event['event'] ?? 'unknown'),
            json_encode($event, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES)
        ]);

        // Basic reconciliation examples: update transfer / payment_link by referenceId
        $ref = (string)($event['referenceId'] ?? $event['reference_id'] ?? '');
        $status = (string)($event['status'] ?? '');

        if ($ref !== '' && $status !== '') {
            // Try transfers first
            DB::exec("UPDATE transfers SET status=?, updated_at=NOW() WHERE id=? AND provider='openpass'", [$status, (int)$ref]);
            DB::exec("UPDATE payment_links SET status=?, updated_at=NOW() WHERE id=? AND provider='openpass'", [$status, (int)$ref]);
        }

        Logger::info('Webhook received', ['provider' => 'openpass', 'type' => $event['type'] ?? null]);

        Response::json(['ok' => true]);
    }
}
